-- #####   OCI Grundfunktionen   #####

-- OCI: Absprung-URL generieren
CREATE OR REPLACE FUNCTION x_10_interfaces.oci__absprung_url__generate_by_krz(
      IN _krz varchar   -- Lieferantenkürzel
    )
  RETURNS varchar AS $$
  DECLARE
    base_url     varchar;
    query_params varchar := '';
    param        record;
  BEGIN
    -- OCI-Basis-URL (HOOK Parameter)
    base_url := oci_value FROM x_10_interfaces.oci_parameter
    WHERE oci_krz = _krz
      AND oci_type = 'HOOK'
      AND oci_name = 'HOOK_URL'
      AND oci_is_active = true;

    IF base_url IS NULL THEN
      RAISE EXCEPTION 'Keine aktive HOOK_URL für Lieferant % gefunden', _krz;
    END IF;

    -- OCI-Parameter zusammenbauen (LOGIN und CONSTANT Parameter)
    FOR param IN
      SELECT oci_name, oci_value
        FROM x_10_interfaces.oci_parameter
       WHERE oci_krz = _krz
         AND oci_type IN ('LOGIN', 'CONSTANT')
         AND oci_is_active = true
         AND coalesce(oci_name, '') <> ''
         AND coalesce(oci_value, '') <> ''
    LOOP
      query_params := query_params || '&' || urlencode(param.oci_name) || '=' || urlencode(param.oci_value);
    END LOOP;

    IF query_params <> '' THEN
        query_params := SUBSTRING(query_params FROM 2); -- entferne führendes '&'
    END IF;

    RETURN base_url || '?' || query_params;
  END $$ LANGUAGE plpgsql;
--

-- OCI: Allgemeine URL-Generator-Funktion für spezifische OCI-Funktionen
CREATE OR REPLACE FUNCTION x_10_interfaces.oci__function_url__generate_by_krz(
      IN _krz          varchar,              -- Lieferantenkürzel
      IN _function     varchar,              -- Funktionstyp (z.B. 'BACKGROUND_SEARCH', 'DETAIL', 'VALIDATE')
      IN _searchstring varchar DEFAULT null, -- Optional: Suchstring
      IN _productid    varchar DEFAULT null, -- Optional: Produkt-ID
      IN _quantity     numeric DEFAULT null  -- Optional: Menge
    )
  RETURNS varchar AS $$
  DECLARE
    base_url     varchar;
    query_params varchar;
  BEGIN
    -- Basis-URL generieren
    base_url := x_10_interfaces.oci__absprung_url__generate_by_krz(_krz);

    -- Funktionsspezifische Parameter hinzufügen
    query_params := '&FUNCTION=' || _function;
    CASE _function
      WHEN 'BACKGROUND_SEARCH' THEN
        IF _searchstring IS NULL THEN
          RAISE EXCEPTION 'Parameter searchstring darf nicht NULL sein für BACKGROUND_SEARCH';
        END IF;
        query_params := query_params || '&SEARCHSTRING=' || urlencode(_searchstring);
      WHEN 'DETAIL' THEN
        IF _productid IS NULL THEN
          RAISE EXCEPTION 'Parameter productid darf nicht NULL sein für DETAIL';
        END IF;
        query_params := query_params || '&PRODUCTID=' || urlencode(_productid);
      WHEN 'VALIDATE' THEN
        IF _productid IS NULL OR _quantity IS NULL THEN
          RAISE EXCEPTION 'Parameter productid und quantity dürfen nicht NULL sein für VALIDATE';
        END IF;
        query_params := query_params || '&PRODUCTID=' || urlencode(_productid) || '&QUANTITY=' || _quantity;
      ELSE
        RAISE EXCEPTION 'Unbekannte Funktion: %. Erlaubte Werte sind: BACKGROUND_SEARCH, DETAIL, VALIDATE', _function;
    END CASE;

    -- Vollständige URL zurückgeben
    RETURN base_url || query_params;
  END $$ LANGUAGE plpgsql;
--

-- OCI: Response in JSON umwandeln
CREATE OR REPLACE FUNCTION x_10_interfaces.oci__convert__html_response_to_json(
      _html_content text
    )
  RETURNS jsonb AS $$
  DECLARE
    result        jsonb := '[]'::jsonb;                     -- Ergebnis-JSONB
    lines         text[];                                   -- Array von Zeilen des HTML-Inputs
    line          text;                                     -- Einzelne Zeile aus dem Array
    key           text;                                     -- Extrahierter Schlüssel aus der HTML-Zeile
    value         text;                                     -- Extrahierter Wert aus der HTML-Zeile
    index         integer;                                  -- Aktueller Index als Text
    current_item  jsonb := '{}'::jsonb;                     -- Aktuelles JSON-Objekt für einen Artikel
    current_index integer := 0;                             -- Aktueller Artikelindex als Integer
    pattern       text := 'name="([^"]+)" value="([^"]*)"'; -- RegEx-Muster für das Parsen von Schlüssel-Wert-Paaren
  BEGIN
    -- HTML in Zeilen zerlegen
    lines := regexp_split_to_array(_html_content, E'\n');

    -- Jede Zeile verarbeiten
    FOREACH line IN ARRAY lines LOOP
        -- Überprüfen, ob die Zeile einen Input-Wert enthält
        IF line ~ pattern THEN
            -- Schlüssel und Wert aus der Zeile extrahieren
            raise notice 'line: %', line;
            SELECT (regexp_matches(line, pattern))[1] INTO key;
            SELECT (regexp_matches(line, pattern))[2] INTO value;

            -- Index des Artikels extrahieren, sicherstellen, dass das Ergebnis nicht NULL ist
            SELECT (regexp_matches(key, '\\[(\\d+)\\]'))[1] INTO index;

            -- Falls Index leer, dann Index bei Langtext anders bestimmen
            IF index IS null THEN
              SELECT (regexp_matches(key, '_(\\d+):132\\[\\]'))[1] INTO index;
            END IF;

            IF index IS NOT null THEN
                -- Neues Objekt starten, wenn der Index wechselt
                IF current_index <> index THEN
                    -- Vorheriges Objekt zum Ergebnis hinzufügen, wenn es nicht leer ist
                    IF current_index <> 0 THEN
                        result := result || jsonb_build_array(current_item);
                    END IF;
                    current_item := '{}'::jsonb;  -- Neues JSON-Objekt für neuen Artikel starten
                    current_index := index;       -- Index aktualisieren
                END IF;

                -- Schlüssel ohne Index für das JSON-Objekt extrahieren
                SELECT regexp_replace(key, E'\\[\\d+\\]', '') INTO key;
                -- Wert dem aktuellen Artikelobjekt hinzufügen
                current_item := jsonb_set(current_item, ARRAY[key], to_jsonb(value), true);
            END IF;
        END IF;
    END LOOP;

    -- Letzten Artikel zum Ergebnis hinzufügen, wenn vorhanden
    IF current_index <> 0 THEN
        result := result || jsonb_build_array(current_item);
    END IF;

    RETURN result;
  END $$ LANGUAGE plpgsql;
--
